package com.squarepeghires.githubing.github;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.squarepeghires.githubing.git.GitCommit;
import com.squarepeghires.githubing.git.GitCommitStatus;
import java.util.List;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.codec.json.Jackson2JsonDecoder;
import org.springframework.stereotype.Service;
import org.springframework.web.reactive.function.client.WebClient;

@Service
public class GitHubClient {

    private final WebClient webClient;

    public GitHubClient(
            @Value("${github.baseUrl}") String baseurl, @Value("${github.accessToken}") String accessToken) {
        this.webClient = createWebClient(baseurl, accessToken);
    }

    private static WebClient createWebClient(String baseurl, String accessToken) {
        return WebClient.builder()
                .baseUrl(baseurl)
                .defaultHeaders(httpHeaders -> {
                    if (!accessToken.isEmpty()) {
                        httpHeaders.setBearerAuth(accessToken);
                    }
                    httpHeaders.setAccept(List.of(new MediaType("application", "vnd.github+json")));
                })
                .codecs(configurer -> configurer
                        .defaultCodecs()
                        .jackson2JsonDecoder(new Jackson2JsonDecoder(createObjectMapper(), MediaType.APPLICATION_JSON)))
                .build();
    }

    private static ObjectMapper createObjectMapper() {
        ObjectMapper mapper = new ObjectMapper()
                .findAndRegisterModules()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        SimpleModule module = new SimpleModule();
        module.addDeserializer(GitCommitStatus.class, new GitHubCommitStatusDeserializer());
        mapper.registerModule(module);

        return mapper;
    }

    public List<GitCommit> getCommits(String owner, String repo, int limit) {
        ResponseEntity<List<GitCommit>> response = webClient
                .get()
                .uri(uriBuilder -> uriBuilder
                        .path("/repos/{owner}/{repo}/commits")
                        .queryParam("per_page", limit)
                        .build(owner, repo))
                .retrieve()
                .toEntityList(GitCommit.class)
                .block();

        assert response != null;

        return response.getBody();
    }

    public GitCommitStatus getStatus(String owner, String repo, String sha) {
        ResponseEntity<GitCommitStatus> response = webClient
                .get()
                .uri(uriBuilder -> uriBuilder
                        .path("/repos/{owner}/{repo}/commits/{ref}/status")
                        .build(owner, repo, sha))
                .retrieve()
                .toEntity(GitCommitStatus.class)
                .block();

        assert response != null;

        return response.getBody();
    }
}
