package com.squarepeghires.githubing.github;

import static com.squarepeghires.githubing.github.GitHubAssertions.assertCommits;
import static org.junit.jupiter.api.Assertions.*;

import com.squarepeghires.githubing.git.GitCommit;
import com.squarepeghires.githubing.git.GitCommitStatus;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.List;
import okhttp3.mockwebserver.MockResponse;
import okhttp3.mockwebserver.MockWebServer;
import okhttp3.mockwebserver.RecordedRequest;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

class GitHubClientTest {
    private MockWebServer mockApi;
    private GitHubClient client;

    @BeforeEach
    void setUp() throws IOException {
        mockApi = new MockWebServer();
        mockApi.start();
        client = new GitHubClient(mockApi.url("").toString(), "mockToken");
    }

    @AfterEach
    void tearDown() throws IOException {
        mockApi.shutdown();
    }

    @Test
    void getCommits() throws Exception {
        InputStream mockResponse = this.getClass().getResourceAsStream("commits.json");
        assertNotNull(mockResponse);

        mockApi.enqueue(new MockResponse()
                .setResponseCode(200)
                .setBody(new String(mockResponse.readAllBytes(), StandardCharsets.UTF_8))
                .addHeader("Content-Type", "application/json"));

        List<GitCommit> commits = client.getCommits("owner", "repo", 7);

        assertCommits(commits);

        RecordedRequest request = mockApi.takeRequest();

        assertEquals("GET", request.getMethod());
        assertEquals("Bearer mockToken", request.getHeader("Authorization"));
        assertEquals("application/vnd.github+json", request.getHeader("Accept"));
        assertEquals("/repos/owner/repo/commits?per_page=7", request.getPath());
    }

    @Test
    void getStatus() throws Exception {
        InputStream mockResponse = this.getClass().getResourceAsStream("status.json");
        assertNotNull(mockResponse);

        mockApi.enqueue(new MockResponse()
                .setResponseCode(200)
                .setBody(new String(mockResponse.readAllBytes(), StandardCharsets.UTF_8))
                .addHeader("Content-Type", "application/json"));

        GitCommitStatus status = client.getStatus("owner", "repo", "mysha");

        assertEquals(GitCommitStatus.SUCCESS, status);

        RecordedRequest request = mockApi.takeRequest();

        assertEquals("GET", request.getMethod());
        assertEquals("Bearer mockToken", request.getHeader("Authorization"));
        assertEquals("application/vnd.github+json", request.getHeader("Accept"));
        assertEquals("/repos/owner/repo/commits/mysha/status", request.getPath());
    }
}
