package com.squarepeghires.githubing.github;

import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import com.squarepeghires.githubing.git.GitCommit;
import com.squarepeghires.githubing.git.GitCommitStatus;
import com.squarepeghires.githubing.git.GitUser;
import java.io.InputStream;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.List;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.test.web.servlet.MockMvc;

@SpringBootTest
@AutoConfigureMockMvc
class GitHubControllerTest {
    @Autowired
    private MockMvc mockMvc;

    @MockBean
    private GitHubClient gitHubClient;

    @Test
    void lastCommits() throws Exception {
        var commit = new GitCommit(
                new URL("https://github.com/commit"),
                "dummySha",
                new GitUser("username", new URL("https://github.com/user"), new URL("https://github.com/avatar")));
        commit.setMessage("commit message");
        commit.setDate(new Date(1367075400000L).toInstant());

        when(gitHubClient.getCommits("microsoft", "TypeScript", 25)).thenReturn(List.of(commit));

        InputStream expected = this.getClass().getResourceAsStream("apiCommits.json");
        assertNotNull(expected);

        this.mockMvc
                .perform(get("/api/commits/typescript"))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(content().json(new String(expected.readAllBytes(), StandardCharsets.UTF_8)));
    }

    @Test
    void commitStatus() throws Exception {
        when(gitHubClient.getStatus("microsoft", "TypeScript", "mySha")).thenReturn(GitCommitStatus.PENDING);

        this.mockMvc
                .perform(get("/api/commits/typescript/mySha"))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(content().string("\"PENDING\""));
    }
}
