import { TestBed } from '@angular/core/testing'
import { CommitService } from './commit.service'
import {
    HttpClientTestingModule,
    HttpTestingController,
} from '@angular/common/http/testing'
import { HttpClient } from '@angular/common/http'
import { GitCommitStatus } from './model'
import { commitMock, commitRawMock } from '../test'

describe('CommitService', () => {
    let httpTestingController: HttpTestingController
    let service: CommitService

    beforeEach(() => {
        TestBed.configureTestingModule({
            imports: [HttpClientTestingModule],
        })
        TestBed.inject(HttpClient)
        httpTestingController = TestBed.inject(HttpTestingController)
        service = TestBed.inject(CommitService)
    })

    afterEach(() => {
        httpTestingController.verify()
    })

    it('should be created', () => {
        expect(service).toBeTruthy()
    })

    it('getCommit should return api results', () => {
        service
            .getCommits()
            .subscribe((response) => expect(response).toEqual([commitMock]))
        const request = httpTestingController.expectOne(
            'https://githubing.squarepeghires.com/api/commits/typescript',
        )
        expect(request.request.method).toEqual('GET')
        request.flush([commitRawMock])
    })

    it('getCommit should group results by authors and sort by date', () => {
        const commitMock1 = {
            ...commitMock,
            author: { ...commitMock.author, login: 'AAA' },
            date: '2013-04-27T17:10:01.000+02:00',
        }
        const commitMock2 = {
            ...commitMock,
            author: { ...commitMock.author, login: 'BBB' },
            date: '2013-04-27T17:10:02.000+02:00',
        }
        const commitMock3 = {
            ...commitMock,
            author: { ...commitMock.author, login: 'AAA' },
            date: '2013-04-27T17:10:03.000+02:00',
        }

        const expected = [commitMock3, commitMock1, commitMock2].map(
            (commitRaw) => ({ ...commitRaw, date: new Date(commitRaw.date) }),
        )

        service
            .getCommits()
            .subscribe((response) => expect(response).toEqual(expected))
        const request = httpTestingController.expectOne(
            'https://githubing.squarepeghires.com/api/commits/typescript',
        )
        expect(request.request.method).toEqual('GET')
        request.flush([commitMock1, commitMock2, commitMock3])
    })

    it('getStatus should return api result', () => {
        service
            .getStatus(commitMock)
            .subscribe((response) =>
                expect(response).toEqual(GitCommitStatus.Success),
            )
        const request = httpTestingController.expectOne(
            'https://githubing.squarepeghires.com/api/commits/typescript/mySha',
        )
        expect(request.request.method).toEqual('GET')
        request.flush(GitCommitStatus.Success)
    })
})
