import { Injectable } from '@angular/core'
import { HttpClient } from '@angular/common/http'
import { GitCommit, GitCommitStatus } from './model'
import { catchError, map, Observable, of } from 'rxjs'
import { environment } from '../environments/environment'
import { formatDistanceToNow } from 'date-fns'

@Injectable({
    providedIn: 'root',
})
export class CommitService {
    private failedBefore = false
    private commits: GitCommit[] | undefined

    constructor(private http: HttpClient) {}

    getCommits(): Observable<GitCommit[]> {
        return this.commits ? of(this.commits) : this.fetchCommits()
    }

    fetchCommits(): Observable<GitCommit[]> {
        const response = this.get('/api/commits/typescript', []).pipe(
            map((commits) =>
                commits
                    .map((commit) => {
                        const date = new Date(commit.date)

                        return {
                            ...commit,
                            date,
                            dateAgo: formatDistanceToNow(date, {
                                addSuffix: true,
                            }),
                        }
                    })
                    .sort(
                        (a, b) =>
                            a.author.login.localeCompare(b.author.login) ||
                            b.date.getTime() - a.date.getTime(),
                    ),
            ),
        )
        response.subscribe((commits) => (this.commits = commits))

        return response
    }

    getStatus(commit: GitCommit): Observable<GitCommitStatus | undefined> {
        return this.get(`/api/commits/typescript/${commit.sha}`, undefined)
    }

    private get(urlPath: string, fallback): Observable<any> {
        return this.http.get(`${environment.backendBaseUrl}${urlPath}`).pipe(
            catchError(() => {
                if (!this.failedBefore) {
                    alert('Failed API call.')
                }
                this.failedBefore = true

                return of(fallback)
            }),
        )
    }
}
